document.addEventListener("DOMContentLoaded", function() {
    
    // Gestion du bouton Switch (Mode Artistique)
    const modeSwitch = document.getElementById('switch-sleep-awake');
    if(modeSwitch) {
        modeSwitch.addEventListener('change', function() {
            if (this.checked) window.location.href = 'artistique.html'; 
            else window.location.href = 'carte.html';
        });
    }

    // --- VARIABLES ---
    window.listeIncendies = []; 
    var chartAnneesInstance = null;
    var chartDetailsInstance = null;
    var chartTendanceInstance = null; 
    
    var geoJsonLayer = null; 
    var layerSelectionne = null;
    var currentCityName = null;

    // Initialisation du groupe Canicule
    window.layerCanicule = L.layerGroup(); 

    // --- SYSTEME D'AIDE ---
    const modal = document.getElementById('explanation-modal');
    const explanationText = document.getElementById('explanation-text');
    const modalTitle = document.getElementById('modal-title');
    const closeModal = document.querySelector('.close-modal');

    window.ouvrirExplication = function(element) {
        if(window.event) { window.event.stopPropagation(); window.event.preventDefault(); }
        const texte = element.getAttribute('data-explanation');
        const titre = element.getAttribute('data-title');
        if(explanationText) explanationText.innerHTML = texte;
        if(modalTitle) modalTitle.textContent = titre;
        if(modal) modal.style.display = 'block';
    };

    if(closeModal) closeModal.addEventListener('click', () => { modal.style.display = 'none'; });
    window.addEventListener('click', (e) => { if (e.target === modal) modal.style.display = 'none'; });

    // --- ACCORDÉON ---
    function initAccordion() {
        const headers = document.querySelectorAll('.accordion-header');
        const allItems = document.querySelectorAll('.accordion-item'); 
        headers.forEach(header => {
            header.addEventListener('click', (e) => {
                if(e.target.closest('button') || e.target.classList.contains('aide-data')) return;
                const item = header.parentElement;
                const estDejaOuvert = item.classList.contains('ouvert');
                allItems.forEach(el => el.classList.remove('ouvert'));
                if (!estDejaOuvert) item.classList.add('ouvert');
            });
        });
    }
    initAccordion();

    // --- CARTE LEAFLET ---
    window.map = L.map('map').setView([44.8378, -0.5795], 11);
    L.tileLayer('https://{s}.basemaps.cartocdn.com/dark_all/{z}/{x}/{y}{r}.png', {
        attribution: '&copy; OSM &copy; CARTO', maxZoom: 19
    }).addTo(window.map);

    // --- CHARGEMENT DONNÉES ---
    fetch('incendie.json?v=' + Date.now()).then(r => r.json()).then(data => {
        window.listeIncendies = data;
        initFiltersFromData();
        chargerGeoJSON();
    });

    function chargerGeoJSON() {
        fetch('commune.geojson').then(r => r.json()).then(geoData => {
            geoJsonLayer = L.geoJSON(geoData, {
                style: function(f) {
                    var surf = calculerSurfaceTotaleVille(f.properties.nom_com || f.properties.nom);
                    return { 
                        fillColor: choisirCouleur(surf), 
                        weight: 1, 
                        opacity: 1, 
                        color: '#333', 
                        fillOpacity: surf > 0 ? 0.7 : 0.2 
                    };
                },
                onEachFeature: function(feature, layer) {
                    var nom = feature.properties.nom_com || feature.properties.nom;

                    layer.bindTooltip(nom, { 
                        permanent: false, 
                        direction: 'center', 
                        className: 'label-commune' 
                    });
                    
                    layer.on({
                        mouseover: function(e) {
                            var l = e.target;
                            if (l !== layerSelectionne) {
                                l.setStyle({ weight: 3, color: '#ffffff', opacity: 1, fillOpacity: 0.9 });
                                l.bringToFront();
                            }
                        },
                        mouseout: function(e) {
                            var l = e.target;
                            if (l !== layerSelectionne) {
                                geoJsonLayer.resetStyle(l); 
                            }
                        },
                        click: function(e) { 
                            if (document.getElementById('btn-canicule') && document.getElementById('btn-canicule').classList.contains('actif')) return;
                            
                            if (layerSelectionne) geoJsonLayer.resetStyle(layerSelectionne);
                            layerSelectionne = layer;
                            
                            layer.setStyle({ weight: 4, color: '#ff4500', fillOpacity: 1 });
                            layer.bringToFront();

                            window.map.flyTo(e.latlng, 12, { duration: 1.5 });
                            
                            currentCityName = nom;
                            
                            updateGraphique1UI(nom);
                            updateGraphique2UI(nom);
                            updateGraphiqueTendanceUI(nom); 
                        }
                    });
                }
            }).addTo(window.map);
        });
    }
    
    // --- OUTILS ---
    function nettoyerNom(t) { return String(t || "").toUpperCase().normalize("NFD").replace(/[\u0300-\u036f]/g, "").replace(/-/g, " ").replace(/^ST /, "SAINT ").trim(); }
    function lireNombre(v) { return v ? Number(String(v).replace(',', '.')) : 0; }
    function choisirCouleur(s) { return s > 100 ? '#4a0000' : (s > 50 ? '#8b0000' : (s > 10 ? '#ff0000' : (s > 1 ? '#ff4500' : (s > 0 ? '#ffa500' : '#2d5a3d')))); }
    
    function calculerSurfaceTotaleVille(nom) {
        let t = 0; const n = nettoyerNom(nom);
        window.listeIncendies.filter(l => nettoyerNom(l["Nom de la commune"]) === n).forEach(l => t += lireNombre(l["Surface parcourue (m2)"]));
        return Math.round(t / 10000 * 100) / 100;
    }

    // --- GRAPH 1  ---
    function updateGraphique1UI(nomVille) {
        var titre = document.getElementById('nom-ville'); if(titre) titre.textContent = nomVille;
        const s = parseInt(document.getElementById('filter-start').value);
        const e = parseInt(document.getElementById('filter-end').value);
        const n = nettoyerNom(nomVille);
        
        const feux = window.listeIncendies.filter(l => { const a = parseInt(l["Année"]); return nettoyerNom(l["Nom de la commune"]) === n && a >= s && a <= e; });
        
        let total = 0; const parAn = {};
        for(let y=s; y<=e; y++) parAn[y]=0;
        feux.forEach(f => { 
            const surf = lireNombre(f["Surface parcourue (m2)"]); 
            total += surf; 
            parAn[parseInt(f["Année"])] += surf; 
        });

        document.getElementById('stat-surface').textContent = (total/10000).toFixed(2) + ' ha';
        
        var elNb = document.getElementById('stat-nb');
        if(elNb) elNb.textContent = feux.length;

        var elAnneeMax = document.getElementById('stat-annee-max');
        let maxS = 0; let aMax = "-";
        for(const [an, val] of Object.entries(parAn)) {
            if(val > maxS) { maxS = val; aMax = an; }
        }
        if(elAnneeMax) elAnneeMax.textContent = aMax;

        var ctx = document.getElementById('graphiqueAnnees').getContext('2d');
        if (chartAnneesInstance) chartAnneesInstance.destroy();
        chartAnneesInstance = new Chart(ctx, {
            type: 'line',
            data: {
                labels: Object.keys(parAn),
                datasets: [{ label: 'Surface (ha)', data: Object.values(parAn).map(v=>v/10000), borderColor: '#ff4500', backgroundColor: 'rgba(255,69,0,0.2)', fill: true }]
            },
            options: { responsive: true, maintainAspectRatio: false, plugins: { legend: {display:false} }, scales: { x: { grid:{display:false} }, y: { grid:{color:'#444'} } } }
        });
    }

    // --- GRAPH 2 : CAUSES ---
   
    function updateGraphique2UI(nomVille) {
        const s = parseInt(document.getElementById('filter-start-2').value);
        const e = parseInt(document.getElementById('filter-end-2').value);
        const n = nettoyerNom(nomVille);
        
        const feux = window.listeIncendies.filter(l => { const a = parseInt(l["Année"]); return nettoyerNom(l["Nom de la commune"]) === n && a >= s && a <= e; });
        
        // On initialise bien le compteur 'Inconnue'
        const causes = {};
        for(let y=s; y<=e; y++) causes[y] = { 'Inconnue':0, 'Accidentelle':0, 'Malveillance':0, 'Naturelle':0 };

        feux.forEach(f => {
            let nat = (f["Nature"]||"").toLowerCase();
            let cat = 'Inconnue'; // Par défaut, c'est inconnue
            
            // Logique de détection
            if(nat.includes('foudre')) cat = 'Naturelle';
            else if(nat.includes('malveillance')||nat.includes('incendiaire')) cat = 'Malveillance';
            else if(nat.includes('travaux')||nat.includes('imprudence')) cat = 'Accidentelle';
            
            // Si aucune condition au-dessus n'est remplie, cat reste 'Inconnue'
            causes[parseInt(f["Année"])][cat]++;
        });

        var ctx = document.getElementById('graphiqueDetails').getContext('2d');
        if (chartDetailsInstance) chartDetailsInstance.destroy();
        
        const labels = Object.keys(causes);
        chartDetailsInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: labels,
                datasets: [
                    { 
                        label: 'Malveillance', 
                        data: labels.map(y=>causes[y].Malveillance), 
                        backgroundColor: '#d63031' 
                    },
                    { 
                        label: 'Accident', 
                        data: labels.map(y=>causes[y].Accidentelle), 
                        backgroundColor: '#e67e22' 
                    },
                    { 
                        label: 'Naturelle', 
                        data: labels.map(y=>causes[y].Naturelle), 
                        backgroundColor: '#27ae60' 
                    },
        
                    { 
                        label: 'Inconnue', 
                        data: labels.map(y=>causes[y].Inconnue), 
                        backgroundColor: '#95a5a6' // Gris neutre
                    }
                ]
            },
            options: { 
                responsive: true, 
                maintainAspectRatio: false, 
                plugins: {
                    legend: { labels: { color: '#ccc' } } 
                },
                scales: { 
                    x: { stacked: true, grid:{display:false}, ticks: { color: '#ccc'} }, 
                    y: { stacked: true, grid:{color:'#444'}, ticks: { color: '#ccc'} } 
                } 
            }
        });
    }
    // --- GRAPH 3  ---
    function updateGraphiqueTendanceUI(nomVille) {
        const n = nettoyerNom(nomVille);
        const feux = window.listeIncendies.filter(l => nettoyerNom(l["Nom de la commune"]) === n);
        
        // 1. Récupération des éléments HTML
        const elMoy = document.getElementById('stat-tendance-moyenne');
        const elEcart = document.getElementById('stat-tendance-ecarttype');
        const elCorr = document.getElementById('stat-tendance-correlation');
        const elInterp = document.getElementById('stat-tendance-interpretation');
        const elR2 = document.getElementById('stat-tendance-r2');
        var ctx = document.getElementById('graphiqueTendance').getContext('2d');

        
        const annees = feux.map(f => parseInt(f["Année"])).filter(a => !isNaN(a));
        
        
        if (feux.length === 0 || annees.length === 0) {
            
           
            if(elMoy) elMoy.textContent = "-";
            if(elEcart) elEcart.textContent = "-";
            if(elR2) elR2.textContent = "-";
            if(elCorr) { elCorr.textContent = "-"; elCorr.style.color = "#ccc"; }
            if(elInterp) { elInterp.textContent = "Aucun historique"; elInterp.style.color = "#ccc"; }

            
            if (chartTendanceInstance) chartTendanceInstance.destroy();
            chartTendanceInstance = new Chart(ctx, {
                type: 'bar',
                data: { labels: [], datasets: [] },
                options: { 
                    responsive: true, maintainAspectRatio: false,
                    plugins: { 
                        legend: { display: false },
                        title: { display: true, text: "Aucun incendie enregistré", color: "#888" }
                    },
                    scales: { x: { display: false }, y: { display: false } }
                }
            });
            return; 
        }

        //dans le cas où les données n'existent pas
        const parAn = {};
        const minY = Math.min(...annees);
        const maxY = Math.max(...annees);
        for(let y=minY; y<=maxY; y++) parAn[y] = 0;
        
        feux.forEach(f => {
            let a = parseInt(f["Année"]);
            if(!isNaN(a)) parAn[a] += lireNombre(f["Surface parcourue (m2)"]) / 10000;
        });

        const xData = [];
        const yData = [];
        for(let y=minY; y<=maxY; y++) {
            xData.push(y);
            yData.push(parAn[y]);
        }

        // Calculs Statistiques
        const nPoints = xData.length;
        const sumY = yData.reduce((a, b) => a + b, 0);
        const meanY = sumY / nPoints;
        const variance = yData.reduce((sum, val) => sum + Math.pow(val - meanY, 2), 0) / nPoints;
        const stdDev = Math.sqrt(variance);

        const meanX = xData.reduce((a, b) => a + b, 0) / nPoints;
        let num = 0, denX = 0, denY = 0;
        for(let i=0; i<nPoints; i++) {
            let dx = xData[i] - meanX;
            let dy = yData[i] - meanY;
            num += dx * dy;
            denX += dx * dx;
            denY += dy * dy;
        }
        const correlation = (denX === 0 || denY === 0) ? 0 : num / Math.sqrt(denX * denY);

        const sumX = xData.reduce((a, b) => a + b, 0);
        const sumXY = xData.reduce((sum, x, i) => sum + x * yData[i], 0);
        const sumXX = xData.reduce((sum, x) => sum + x * x, 0);
        const slope = (nPoints * sumXY - sumX * sumY) / (nPoints * sumXX - sumX * sumX);
        const intercept = (sumY - slope * sumX) / nPoints;
        const r2 = correlation * correlation; 

        // Affichage HTML
        if(elMoy) elMoy.textContent = meanY.toFixed(2);
        if(elEcart) elEcart.textContent = stdDev.toFixed(2);
        if(elR2) elR2.textContent = (r2 * 100).toFixed(1) + '%';
        
        if(elCorr) {
            elCorr.textContent = correlation.toFixed(2);
            elCorr.style.color = correlation > 0 ? '#d63031' : '#00b894';
        }

        if(elInterp) {
            if(Math.abs(correlation) < 0.2) {
                elInterp.textContent = "Aucune tendance";
                elInterp.style.color = "#ccc";
            } else if (correlation > 0) {
                elInterp.textContent = "Risque en hausse";
                elInterp.style.color = "#d63031";
            } else {
                elInterp.textContent = "Risque en baisse";
                elInterp.style.color = "#00b894";
            }
        }

        const trendData = xData.map(x => slope * x + intercept);
        
        if (chartTendanceInstance) chartTendanceInstance.destroy();

        chartTendanceInstance = new Chart(ctx, {
            type: 'bar',
            data: {
                labels: xData,
                datasets: [
                    {
                        label: 'Surface Réelle',
                        data: yData,
                        backgroundColor: 'rgba(230, 126, 34, 0.4)',
                        order: 2
                    },
                    {
                        label: 'Tendance',
                        data: trendData,
                        borderColor: slope > 0 ? '#d63031' : '#00b894',
                        borderWidth: 2,
                        borderDash: [5, 5],
                        type: 'line',
                        pointRadius: 0,
                        order: 1
                    }
                ]
            },
            options: {
                responsive: true, maintainAspectRatio: false,
                plugins: { legend: { display: true, position:'bottom', labels:{color:'#ccc'} } },
                scales: { 
                    x: { grid: { display: false } },
                    y: { grid: { color: '#444' }, title: { display:true, text:'Hectares' } }
                }
            }
        });
    }

    function initFiltersFromData() {
        const s1 = document.getElementById('filter-start'), e1 = document.getElementById('filter-end');
        const s2 = document.getElementById('filter-start-2'), e2 = document.getElementById('filter-end-2');
        if(!s1) return;
        const annees = window.listeIncendies.map(l => parseInt(l["Année"])).filter(a => !isNaN(a));
        if(annees.length === 0) return;
        const min = Math.min(...annees), max = Math.max(...annees);
        
        const remplir = (sel) => { sel.innerHTML=""; for(let y=min; y<=max; y++) sel.add(new Option(y,y)); };
        [s1, e1, s2, e2].forEach(remplir);
        s1.value = s2.value = min; e1.value = e2.value = max;
        
        const update = () => { if(currentCityName) { updateGraphique1UI(currentCityName); updateGraphique2UI(currentCityName); } };
        s1.onchange = e1.onchange = s2.onchange = e2.onchange = update;
    }

    // Gestion du basculement d'onglets
    const btnIncendie = document.getElementById('btn-incendie');
    const btnCanicule = document.getElementById('btn-canicule');
    if(btnIncendie) btnIncendie.addEventListener('click', () => basculerMode('incendie'));
    if(btnCanicule) btnCanicule.addEventListener('click', () => basculerMode('canicule'));

    function basculerMode(mode) {
        const wrapperIncendie = document.getElementById('mode-incendie-wrapper');
        const wrapperCanicule = document.getElementById('mode-canicule-wrapper');
        const legI = document.getElementById('legende-incendie');
        const legC = document.getElementById('legende-canicule');

        if(mode === 'incendie') {
            btnIncendie.classList.add('actif'); btnCanicule.classList.remove('actif');
            wrapperIncendie.classList.remove('cache'); wrapperCanicule.classList.add('cache');
            if(geoJsonLayer) geoJsonLayer.addTo(window.map);
            if(window.layerCanicule) window.layerCanicule.removeFrom(window.map);
            legI.classList.remove('cache'); legC.classList.add('cache');
        } else {
            btnCanicule.classList.add('actif'); btnIncendie.classList.remove('actif');
            wrapperCanicule.classList.remove('cache'); wrapperIncendie.classList.add('cache');
            if(geoJsonLayer) geoJsonLayer.removeFrom(window.map);
            if(window.layerCanicule) window.layerCanicule.addTo(window.map);
            legI.classList.add('cache'); legC.classList.remove('cache');
        }
    }
    
});